/*
 ************************************************************************************************
 *                                                                                              *
 *  OSA cooperative RTOS for Microchip PIC-controllers: PIC10/12/16/18/24/dsPIC                 *
 *                                                                                              *
 *  URL:        http://wiki.pic24.ru/doku.php/en/osa/ref/intro                                  *
 *              http://picosa.narod.ru                                                          *
 *                                                                                              *
 *----------------------------------------------------------------------------------------------*
 *                                                                                              *
 *  File:       osa_system.h                                                                    *
 *                                                                                              *
 *  Compilers:  HT-PICC STD                                                                     *
 *              HT-PICC18 STD                                                                   *
 *              Microchip C18                                                                   *
 *              Microchip C30                                                                   *
 *                                                                                              *
 *  Programmer: Timofeev Victor                                                                 *
 *              osa@pic24.ru, testerplus@mail.ru                                                *
 *                                                                                              *
 *  Definition: System services                                                                 *
 *                                                                                              *
 *  History:    21.01.2009                                                                      *
 *                                                                                              *
 ************************************************************************************************
 */














/************************************************************************************************
 *                                                                                              *
 *                                      S H C E D U L E R                                       *
 *                                                                                              *
 *                      !!!(Sheduler for MCC30 is in "port\osa_mcc30.h")!!!                     *
 *                                                                                              *
 ************************************************************************************************/


//------------------------------------------------------------------------------
#ifndef OS_Sched
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
#ifdef OS_ENABLE_CRITICAL_SECTION

    #define SKIP_IF_IS_IN_CRITICAL_SECTION()    if (OS_IsInCriticalSection()) goto SCHED_CHECK_READY;
    #define LABEL_SCHED_CHECK_READY             SCHED_CHECK_READY:
    #define NOT_IN_CRITICAL_SECTION()           !OS_IsInCriticalSection()

#else

    #define SKIP_IF_IS_IN_CRITICAL_SECTION()
    #define LABEL_SCHED_CHECK_READY
    #define NOT_IN_CRITICAL_SECTION()           1

#endif
//------------------------------------------------------------------------------




#define IF_NOT_TASK_ENABLE_CONTINUE()   if (!_OS_bTaskEnable || _OS_bTaskPaused) continue;

#define LABEL_SCHED_CONTINUE

#define IF_NOT_TASK_READY()             if (!_OS_bTaskReady)

#define IF_TASK_READY()                 if (_OS_bTaskReady)

#define LABEL_SCHED_SKIP_RUN


#ifdef OS_ENABLE_TTIMERS

    #define IF_NOT_OS_CHECK_DELAYS_GOTO_SCHED_CONTINUE  \
                                        if (_OS_bTaskDelay && !_OS_bTaskCanContinue) continue;
#else

    #define IF_NOT_OS_CHECK_DELAYS_GOTO_SCHED_CONTINUE

#endif



//------------------------------------------------------------------------------
#if defined(_OS_CUR_FLAGS_IN_OS_state)
//------------------------------------------------------------------------------

    #define _OS_SET_STATE()     OS_state = _OS_CurTask->State
    #define _OS_RESTORE_INDF()

//------------------------------------------------------------------------------
#else
//------------------------------------------------------------------------------

    #define _OS_SET_STATE()     _OS_SET_FSR_CUR_TASK()
    #define _OS_RESTORE_INDF()  _OS_SET_FSR_CUR_TASK()

//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------




//------------------------------------------------------------------------------
#if !defined(OS_DISABLE_PRIORITY)
//------------------------------------------------------------------------------

    /********************************************************************************************
     *                                                                                          *
     *     P R I O R I T Y   M O D E   S C H E D U L E R                                        *
     *                                                                                          *
     ********************************************************************************************/

    #define OS_IsIdle()     (!OS_IsInCriticalSection() && !OS_Flags.bBestTaskFound)

    #define OS_Sched()                                                                          \
    {                                                                                           \
                                                                                                \
        /* First we suppose that there is no ready task*/                                       \
        OS_Flags.bBestTaskFound = 0;                                                            \
                                                                                                \
        /* MODE: Search of best task                  */                                        \
        OS_Flags.bCheckingTasks = 1;                                                            \
                                                                                                \
        /* Worst of lowest priority                   */                                        \
        _OS_best_priority = OS_WORST_PRIORITY + 1;                                              \
                                                                                                \
        SKIP_IF_IS_IN_CRITICAL_SECTION();                                                       \
                                                                                                \
        /* List all tasks starting from last executed   */                                      \
        _OS_CurTask = _OS_LastTask;                                                             \
                                                                                                \
        do                                                                                      \
        {                                                                                       \
            if (_OS_CurTask >= (OST_TASK_POINTER) &OS_TaskVars[OS_TASKS-1])                     \
                _OS_CurTask = (OST_TASK_POINTER) &OS_TaskVars[0];                               \
            else                                                                                \
                _OS_CurTask++;                                                                  \
                                                                                                \
    LABEL_SCHED_CHECK_READY                                                                     \
                                                                                                \
            _OS_SET_STATE(); /* For 16  MCC18 state is in INDF, for 12, PICC18 - in OS_state*/ \
                                                                                                \
            IF_NOT_TASK_ENABLE_CONTINUE();                           /* Skip empty descriptor */\
                                                                                                \
            if (_OS_best_priority <= (_OS_temp = _OS_cTaskPriority)) /* see p.1 below         */\
                continue;                                                                       \
                                                                                                \
            IF_NOT_TASK_READY()                                                                 \
            {                                                                                   \
                /* Task is not ready yet. Check for it has became ready    */                   \
    SCHED_RUN:;                                                                                 \
                _OS_JumpToTask();                                                               \
                _OS_SaveRetAddr();                                                              \
                if (!OS_Flags.bCheckingTasks) goto SCHED_END;                                   \
                _OS_RESTORE_INDF();                                                             \
            }                                                                                   \
                                                                                                \
    LABEL_SCHED_SKIP_RUN                                                                        \
                                                                                                \
            /* Skip if task is delayed              */                                          \
            IF_NOT_OS_CHECK_DELAYS_GOTO_SCHED_CONTINUE;                                         \
                                                                                                \
            /* Compare priority for all ready tasks      */                                     \
            IF_TASK_READY()                                                                     \
            {                                                                                   \
                OS_Flags.bBestTaskFound = 1;                                                    \
                _OS_BestTask = _OS_CurTask;                                                     \
                _OS_best_priority = _OS_cTaskPriority;                                          \
                                                                                                \
                /* If this task has highest priority then stop searching */                     \
                if (_OS_best_priority == OS_BEST_PRIORITY) break;                               \
            }                                                                                   \
                                                                                                \
    LABEL_SCHED_CONTINUE                                                                        \
                                                                                                \
        } while (NOT_IN_CRITICAL_SECTION() && _OS_CurTask != _OS_LastTask);                     \
                                                                                                \
        /* If there is at least one ready task, then execute it */                              \
                                                                                                \
        if (OS_Flags.bBestTaskFound)                                                            \
        {                                                                                       \
            OS_Flags.bCheckingTasks = 0;                /* MODE: executing best task    */      \
            _OS_LastTask = _OS_CurTask = _OS_BestTask;                                          \
            goto SCHED_RUN;                                                                     \
        }                                                                                       \
                                                                                                \
    SCHED_END:;                                                                                 \
                                                                                                \
    }                                                                                           \

    //-------------------------------------------------------------------------------------------
    // Explaining of some moments:
    //
    // 1.  OS_Temp used here to reduce used RAM by 1 byte. Compiler while comparing
    //     _OS_best_priority with (indf&7) generates one temporary variable. We set it manualy,
    //     so compiler will not use local variable.
    //
    //-------------------------------------------------------------------------------------------



//------------------------------------------------------------------------------
#endif      // !OS_DISABLE_PRIORITY
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
#if defined(OS_DISABLE_PRIORITY)
//------------------------------------------------------------------------------

    /********************************************************************************************
     *                                                                                          *
     *     N O N   P R I O R I T Y   M O D E   S C H E D U L E R                                *
     *                                                                                          *
     ********************************************************************************************/

    #define OS_IsIdle()     0

    //-------------------------------------------------------------------------------------------
    //
    // Non priority mode can be used when there is no need to give more preference to
    // some tasks. (For example there are no time rcitical code). This mode allows to reduce ROM
    // and RAM usage and also increases speed of // scheduler work in OS_TASKS times
    //
    //-------------------------------------------------------------------------------------------


    #ifdef OS_ENABLE_CRITICAL_SECTION
        #define IF_NOT_IS_IN_CRITICAL_SECTION()        if (!OS_IsInCriticalSection())
    #else
        #define IF_NOT_IS_IN_CRITICAL_SECTION()
    #endif

    #define OS_Sched()                                                                          \
    {                                                                                           \
        _OS_temp = 0;                                                                           \
                                                                                                \
        IF_NOT_IS_IN_CRITICAL_SECTION()                                                         \
        {                                                                                       \
            if (_OS_CurTask >= (OST_TASK_POINTER) &OS_TaskVars[OS_TASKS-1])                     \
                _OS_CurTask = (OST_TASK_POINTER) &OS_TaskVars[0];                               \
            else                                                                                \
                _OS_CurTask++;                                                                  \
        }                                                                                       \
                                                                                                \
        _OS_SET_STATE(); /* For 16  MCC18 state is in INDF, for 12, PICC18 - in OS_state*/     \
                                                                                                \
        /* Skip empty descriptor              */                                                \
        if (!_OS_bTaskEnable || _OS_bTaskPaused) goto SCHED_CONTINUE;                           \
                                                                                                \
        IF_NOT_OS_CHECK_DELAYS_GOTO_SCHED_CONTINUE;                                             \
                                                                                                \
        _OS_JumpToTask();                                                                       \
        _OS_SaveRetAddr();                                                                      \
                                                                                                \
    SCHED_CONTINUE:;                                                                            \
                                                                                                \
    }

//------------------------------------------------------------------------------
#endif     //   OS_DISABLE_PRIORITY
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
#endif  // ifndef OS_Shed
//------------------------------------------------------------------------------

/************************************************************************************************/









/************************************************************************************************
 *                                                                                              *
 *     C O M M O N   S Y S T E M   S E R V I C E S                                              *
 *                                                                                              *
 ************************************************************************************************/


#define OS_Run()    for(;;) OS_Sched()


/************************************************************************************
 *
 * Give control to scheduler:
 *
 ************************************************************************************/


/************************************************************************/
/* 1. Return with saving context                                        */
/************************************************************************/
#define OS_Yield()                          _OS_GetReturnPoint()

/************************************************************************/
/* 2. Return without saving context                                     */
/*    (internal service)                                                */
/************************************************************************/
#define _OS_Return()                        _OS_ReturnNoSave()

/************************************************************************/
/* 3. Clear bReady and save context without returning                   */
/************************************************************************/
#define _OS_ClearReady()                    _OS_CLEAR_READY()

/************************************************************************/
/* 4. Clear bReady, set bCanContinue, save context without returning    */
/************************************************************************/
#define _OS_ClearReadySetCanContinue()      _OS_CLEAR_READY_SET_CANCONTINUE()











/************************************************************************/
/*                                                                      */
/* Delay current task                                                   */
/*                                                                      */
/************************************************************************/

#define OS_Delay(delaytime)                        \
    {                                              \
        _OS_InitDelay(delaytime);                  \
        OS_Yield();                                \
    }                                              \





/************************************************************************/
/* Check for timeout expired in last service                            */
/************************************************************************/

#define OS_IsTimeout()              OS_Flags.bTimeout

/************************************************************************/
/* Check for errors:                                                    */
/************************************************************************/

//------------------------------------------------------------------------------
// Check for task creation error

#define OS_IsError()                OS_Flags.bError

//------------------------------------------------------------------------------
// Check for event error

#define OS_IsEventError()           OS_Flags.bEventError

/************************************************************************/
/* Check for one of task is in critical section                         */
/************************************************************************/

#define OS_IsInCriticalSection()    OS_Flags.bInCriticalSection









/************************************************************************************************
 *                                                                                              *
 *                                    I N T E R R U P T S                                       *
 *                                                                                              *
 ************************************************************************************************/


//------------------------------------------------------------------------------
// Enable/disable/restore interrupts

//------------------------------------------------------------------------------
#if   defined(__OSAMCC30__)
//------------------------------------------------------------------------------

    extern _OST_INT_TYPE OS_DI();
    extern void OS_RI(_OST_INT_TYPE);
    #define OS_EI()     SR &= 0xFF1F;

//------------------------------------------------------------------------------
#elif defined(__OSAPICC__)
//------------------------------------------------------------------------------

    extern  char OS_DI (void);
    #define OS_RI(cSavedInt)    {if (cSavedInt & 0x80) _gie = 1;}
    #define OS_EI()             { _gie = 1;}

//------------------------------------------------------------------------------
#elif defined(__OSAPICC18__)
//------------------------------------------------------------------------------

    extern char OS_DI (void);
    extern void OS_RI (char);
    #define OS_EI()         { _gie = 1; if (_ipen) _giel = 1;}
    #define OS_EIH()        { _gie = 1; }
    #define OS_EIL()        { _giel = 1; }

//------------------------------------------------------------------------------
#elif defined(__OSAMCC18__)
//------------------------------------------------------------------------------

    extern void _OS_RI (void);
    extern char OS_DI (void);
    #define OS_RI(cSavedInt) {_wreg = cSavedInt; _OS_RI();}
    #define OS_EI()         { GIE = 1; if (IPEN) GIEL = 1;}
    #define OS_EIH()        { GIE = 1; }
    #define OS_EIL()        { GIEL = 1; }

//------------------------------------------------------------------------------
#elif defined(__OSAPICC12__)
//------------------------------------------------------------------------------

    #define OS_DI()     0
    #define OS_RI(temp)

//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------




//------------------------------------------------------------------------------
#ifdef __OSAMCC30__
//------------------------------------------------------------------------------

    #define OS_EnterInt()
    #define OS_LeaveInt()

//------------------------------------------------------------------------------
#else
//------------------------------------------------------------------------------

    /************************************************************************/
    /*                                                                      */
    /* This service must be called in the begin of ISR                      */
    /*                                                                      */
    /************************************************************************/

    #define OS_EnterInt()                                   \
        static _FSR_TYPE   _OS_FSR_Temp;                    \
        _OS_FSR_Temp = _fsr;                                \

    /************************************************************************/
    /*                                                                      */
    /* This service must be called at the end of ISR                        */
    /*                                                                      */
    /************************************************************************/

    #define OS_LeaveInt()   _fsr = _OS_FSR_Temp

#endif







/************************************************************************************************
 *                                                                                              *
 *                                     C O N D I T I O N S                                      *
 *                                                                                              *
 ************************************************************************************************/

//------------------------------------------------------------------------------
#if defined(__OSA16__) || defined(__OSA12__)
//------------------------------------------------------------------------------

    extern bit _OS_CheckEvent (char);

    #define _OS_CHECK_EVENT(event) if (!_OS_CheckEvent(event)) _OS_Return();

//------------------------------------------------------------------------------
#elif defined(__OSAMCC18__)
//------------------------------------------------------------------------------

    extern void _OS_CheckEvent (void);

    #define _OS_CHECK_EVENT(event) {_OS_temp = event; _OS_CheckEvent();}

//------------------------------------------------------------------------------
#else // For C18 and C30
//------------------------------------------------------------------------------

    extern void _OS_CheckEvent (_OST_INT_TYPE);

    #define _OS_CHECK_EVENT(event) _OS_CheckEvent(event)

//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------




/***********************************************************************************************
 *
 * Two internal macros for reducing source code of operation system.
 *
 * !!! FOR AUTHOR: You should be careful with using these macros because of "event"
 * is any expression including modifying FSR.
 *
 ***********************************************************************************************/

/*
 ********************************************************************************
 *                                                                              *
 *   Put task in wait mode till "event" will not became true                    *
 *                                                                              *
 ********************************************************************************
 */

#define _OS_WAIT_EVENT(event)                       \
{                                                   \
    _OS_ClearReady();                               \
    _OS_CHECK_EVENT(event);                         \
}

/*
 ********************************************************************************
 *                                                                              *
 *   Put task in wait mode till "event" will not became true or timeout will not*
 *   expired                                                                    *
 *                                                                              *
 ********************************************************************************
 */

#define _OS_WAIT_EVENT_TIMEOUT(event, timeout)      \
{                                                   \
    _OS_InitDelay(timeout);                         \
    _OS_ClearReadySetCanContinue();                 \
    _OS_CHECK_EVENT(event);                         \
}







/************************************************************************/
/*  Wait for condition                                                  */
/************************************************************************/

#define OS_Cond_Wait(condition)     _OS_WAIT_EVENT(condition)


/************************************************************************/
/*  Wait for condition. Exit if timeout expired.                        */
/************************************************************************/

#define OS_Cond_Wait_TO(condition, timeout)    _OS_WAIT_EVENT_TIMEOUT(condition, timeout)


















