/*
 ************************************************************************************************
 *                                                                                              *
 *  OSA cooperative RTOS for Microchip PIC-controllers: PIC10/12/16/18/24/dsPIC                 *
 *                                                                                              *
 *  URL:        http://wiki.pic24.ru/doku.php/en/osa/ref/intro                                  *
 *              http://picosa.narod.ru                                                          *
 *                                                                                              *
 *----------------------------------------------------------------------------------------------*
 *                                                                                              *
 *  File:       osa_picc12.c                                                                    *
 *                                                                                              *
 *  Compilers:  HT-PICC STD                                                                     *
 *              HT-PICC18 STD                                                                   *
 *              Microchip C18                                                                   *
 *              Microchip C30                                                                   *
 *                                                                                              *
 *  Programmer: Timofeev Victor                                                                 *
 *              osa@pic24.ru, testerplus@mail.ru                                                *
 *                                                                                              *
 *  Definition: PICC12 specific functions overloads                                             *
 *                                                                                              *
 *  History:    21.01.2009                                                                      *
 *                                                                                              *
 ************************************************************************************************
 */




#define OS_Init_ProcSpec()      // Empty macro for PIC12






/*
 ************************************************************************************************
 *                                                                                              *
 *                                   D E L A Y S   I N   T A S K                                *
 *                                                                                              *
 ************************************************************************************************
 */
//------------------------------------------------------------------------------
#ifdef OS_ENABLE_TTIMERS
//------------------------------------------------------------------------------


/*
 ********************************************************************************
 *                                                                              *
 *   void _OS_InitDelay (_OST_TTIMER Delay)                                     *
 *                                                                              *
 *------------------------------------------------------------------------------*
 *                                                                              *
 *  description:    (Internal function called by system kernel)                 *
 *                                                                              *
 *                  Init task timer delay.Set timer variable in task descriptor,*
 *                  sets bit bDelay and clears bit bCanContinue.                *
 *                                                                              *
 *                  _OS_CurTask must point to descriptor.                       *
 *                                                                              *
 *                                                                              *
 *  parameters:     Delay   - time of delay in system ticks                     *
 *                                                                              *
 *  on return:      none                                                        *
 *                                                                              *
 *  note:           This function overloads "osa.c"                             *
 *                                                                              *
 ********************************************************************************
 */

//------------------------------------------------------------------------------
#define _OS_InitDelay_DEFINED
//------------------------------------------------------------------------------
void _OS_InitDelay (_OST_TTIMER Delay)
{

    _OS_bTaskCanContinue = 0;
    _OS_bTaskDelay = 0;         // Clear bDelay to prevent timer changing in interrupt

    if (Delay)
    {

        Delay ^= -1;
        Delay ++;

        #if     OS_TTIMER_SIZE == 1
            asm("   movlw   2   ");
            asm("   addwf   __OS_CurTask, w ");
            _OS_PIC12_AND_1F();
            asm("   movwf   __fsr    ");
            asm("   movf    ?a__OS_InitDelay, w");
            _OS_SET_TASK_IRP();
            asm("   movwf   __indf");
            _OS_SET_TASK_IRP_0();
        #elif   OS_TTIMER_SIZE == 2
            asm("   movlw   2   ");
            asm("   addwf   __OS_CurTask, w ");
            _OS_PIC12_AND_1F();
            asm("   movwf   __fsr    ");
            asm("   movf    ?__OS_InitDelay + 1, w");
            _OS_SET_TASK_IRP();
            asm("   movwf   __indf");

            _OS_SET_TASK_IRP_0();
            asm("   incf    __fsr ");
            asm("   movf    ?__OS_InitDelay + 2, w");
            _OS_SET_TASK_IRP();
            asm("   movwf   __indf");

            _OS_SET_TASK_IRP_0();

        #elif   OS_TTIMER_SIZE == 4
            asm("   movlw   2   ");
            asm("   addwf   __OS_CurTask, w ");
            _OS_PIC12_AND_1F();
            asm("   movwf   __fsr    ");
            asm("   movf    ?__OS_InitDelay + 1, w");
            _OS_SET_TASK_IRP();
            asm("   movwf   __indf");

            _OS_SET_TASK_IRP_0();
            asm("   incf    __fsr ");
            asm("   movf    ?__OS_InitDelay + 2, w");
            _OS_SET_TASK_IRP();
            asm("   movwf   __indf");

            _OS_SET_TASK_IRP_0();
            asm("   incf    __fsr ");
            asm("   movf    ?__OS_InitDelay + 3, w");
            _OS_SET_TASK_IRP();
            asm("   movwf   __indf");

            _OS_SET_TASK_IRP_0();
            asm("   incf    __fsr ");
            asm("   movf    ?__OS_InitDelay + 4, w");
            _OS_SET_TASK_IRP();
            asm("   movwf   __indf");

            _OS_SET_TASK_IRP_0();
        #endif

        _OS_SET_FSR_CUR_TASK();
        _OS_bTaskDelay = 1;
        _OS_bTaskReady = 1;

    }
}


//------------------------------------------------------------------------------
#endif  // #ifdef OS_TIMER
//------------------------------------------------------------------------------














/************************************************************************************************
 *                                                                                              *
 *                            S Y S T E M   F U N C T I O N S                                   *
 *                                                                                              *
 ************************************************************************************************/




/*
 ********************************************************************************
 *                                                                              *
 *   void _OS_Task_Create(_OST_INT_TYPE priority, OST_CODE_POINTER TaskAddr)    *
 *                                                                              *
 *------------------------------------------------------------------------------*
 *                                                                              *
 *   description:   (Internal function called by system kernel from service     *
 *                  OS_Task_Create)                                             *
 *                  Create task in free descriptor.                             *
 *                                                                              *
 *  parameters:     priority - value from 0 (highest) to 7 (lowest)             *
 *                  TaskAddr - pointer to C-function that contains task         *
 *                                                                              *
 *  on return:      check OS_IsError                                            *
 *                                                                              *
 *  note:           This function overloads "osa.c"                             *
 *                                                                              *
 ********************************************************************************
 */


//------------------------------------------------------------------------------
#define _OS_Task_Create_DEFINED
//------------------------------------------------------------------------------

void _OS_Task_Create(_OST_INT_TYPE priority, OST_CODE_POINTER TaskAddr)
{
    OST_TASK_POINTER Task;

    OS_Flags.bError = 0;


    Task = (OST_TASK_POINTER)OS_TaskVars;
    _OS_temp = OS_TASKS;
    do {
        if (!Task->State.bEnable)
        {
            ((OST_TASK_STATE*)&priority)->bEnable = 1;
            ((OST_TASK_STATE*)&priority)->bReady = 1;

            asm("   incf    __OS_Task_Create$Task, w     ");
            _OS_PIC12_AND_1F();
            asm("   movwf   __fsr                        ");
            asm("   movf    __OS_Task_Create$TaskAddr, w      ");
            _OS_SET_TASK_IRP();
            asm("   movwf    __indf ");
            asm("   decf    __fsr   ");
            _OS_SET_TASK_IRP_0();
            asm("   movf    __OS_Task_Create$priority, w      ");   // priority
            _OS_SET_TASK_IRP();
            asm("   movwf    __indf ");
            _OS_SET_TASK_IRP_0();
            if (Task == _OS_CurTask) OS_state = *((OST_TASK_STATE*)&priority);
            OS_Flags.bError = 0;

            return ;

        }
        Task ++;
    } while (_OS_temp--);

    // No free descriptor
    OS_Flags.bError = 1;
    return ;
}









/*
 ********************************************************************************
 *                                                                              *
 *  bit _OS_CheckEvent (char bEvent)                                            *
 *                                                                              *
 *------------------------------------------------------------------------------*
 *                                                                              *
 *  description:    (Internal function called by system kernel througth         *
 *                  waiting services)                                           *
 *                                                                              *
 *                  Check condition of bEvent. Accordint to bEvent value and    *
 *                  current state of bReady, bDelay and bCanContinue flags,     *
 *                  task becames ready to execute or still remains in waiting   *
 *                  mode.                                                       *
 *                                                                              *
 *  parameters:     bEvent - zero, or non-zero condition                        *
 *                                                                              *
 *  on return:      1 - when condition is true, or timeout occured              *
 *                  0 - condition is false and no timeout                       *
 *                                                                              *
 *  note:           This function overloads "osa.c"                             *
 *                                                                              *
 ********************************************************************************
 */


//------------------------------------------------------------------------------
#define _OS_CheckEvent_DEFINED
//------------------------------------------------------------------------------
bit _OS_CheckEvent (char bEvent)
{
    OS_Flags.bTimeout = 0;

    if (bEvent)
    {

        if (_OS_bTaskReady)
        {
            #ifdef OS_ENABLE_TTIMERS
                _OS_bTaskDelay = 0;
            #endif

            return 1;
        }

        _OS_bTaskReady = 1;

    } else _OS_bTaskReady = 0;

    #ifdef OS_ENABLE_TTIMERS

        if (_OS_bTaskTimeout && _OS_bTaskCanContinue)
        {
            OS_Flags.bTimeout = 1;

            return 1;
        }

    #endif

    return 0;
}
















/************************************************************************************************
 *                                                                                              *
 *                     W O R K   W I T H   D Y N A M I C   T I M E R S                          *
 *                                                                                              *
 ************************************************************************************************/

//-----------------------------------------------------------------
#ifdef OS_ENABLE_DTIMERS
//-----------------------------------------------------------------


/*
 ********************************************************************************
 *                                                                              *
 *  void _OS_Dtimer_Create (OST_DTIMER *dtimer)                                 *
 *                                                                              *
 *------------------------------------------------------------------------------*
 *                                                                              *
 *  description:    (Internal function called by system kernel througth         *
 *                  service OS_Dtimer_Create)                                   *
 *                                                                              *
 *                  Add dynamic timer into list of active dynamic timers. This  *
 *                  timer will be added at first position in list.              *
 *                                                                              *
 *                                                                              *
 *  parameters:     dtimer - pointer to dynamic timer.                          *
 *                  (for PICC16 can be allocated only in bank0 and bank1)       *
 *                                                                              *
 *  on return:      none                                                        *
 *                                                                              *
 *  note:           This function overloads "osa.c"                             *
 *                                                                              *
 ********************************************************************************
 */

//------------------------------------------------------------------------------
#define _OS_Dtimer_Create_DEFINED
//------------------------------------------------------------------------------
void _OS_Dtimer_Create (OST_DTIMER *dtimer)
{
    //------------------------------------------------------------------------------
    // dtimer->Flags = _OS_DTimers.Flags;
        #if _BANKBITS_ == 2
        asm("   bcf     __status, 0             ");
        asm("   btfsc   __OS_Dtimer_Create$dtimer, 6  ");
        asm("   bsf     __status, 0             ");
        #endif
        asm("   movf    __OS_Dtimer_Create$dtimer, w  ");
        _OS_PIC12_AND_1F();
        asm("   movwf   __fsr                   ");
        //-------
        _OS_SET_OS_IRP();
        asm("   movf    __OS_DTimers & 0x1F, w  ");
        _OS_SET_OS_IRP_0();
        //-------
        #if _BANKBITS_ >= 1
        asm("   btfsc   __OS_Dtimer_Create$dtimer, 5 ");
        asm("   bsf     __fsr, 5            ");
        #endif
        #if _BANKBITS_ == 2
        asm("   btfsc   __status, 0 ");
        asm("   bsf     __fsr, 6            ");
        #endif
        asm("   movwf   __indf              ");
        _fsr = 0;

    //------------------------------------------------------------------------------
    // dtimer->Next = _OS_DTimers.Next;

        /* Now CARRY = __OS_Dtimer_Create$dtimer.6 */

        asm("   movf    __OS_Dtimer_Create$dtimer, w      ");
        _OS_PIC12_AND_1F();
        asm("   movwf   __fsr                       ");
        asm("   incf    __fsr, f                    ");
        //-------
        _OS_SET_OS_IRP();
        asm("   movf    (__OS_DTimers + 1) & 0x1F, w");
        _OS_SET_OS_IRP_0();
        //-------
    #if _BANKBITS_ >= 1
        asm("   btfsc   __OS_Dtimer_Create$dtimer, 5      ");
        asm("   bsf     __fsr, 5                    ");
    #endif
    #if _BANKBITS_ == 2
        asm("   btfsc   __status, 0 ");
        asm("   bsf     __fsr, 6                    ");
    #endif
        asm("   movwf   __indf                      ");
        _fsr = 0;

    //------------------------------------------------------------------------------
    dtimer->Timer = 0;


    _OS_DTimers.Next = (OST_DTIMER*)dtimer;
    _OS_DTimers.Flags.bNextEnable = 1;

}







/*
 ********************************************************************************
 *                                                                              *
 *  void _OS_Dtimer_Delete (OST_DTIMER *dtimer)                                 *
 *                                                                              *
 *------------------------------------------------------------------------------*
 *                                                                              *
 *  description:    (Internal function called by system kernel througth         *
 *                  service OS_Dtimer_Delete)                                   *
 *                                                                              *
 *                  Delete dynamic timer from list of active timers. It is      *
 *                  recommended to delete all unused dynamic timers to avoid    *
 *                  speed down.                                                 *
 *                                                                              *
 *  parameters:     dtimer - pointer to deleting dynamic timer                  *
 *                                                                              *
 *  on return:      none                                                        *
 *                                                                              *
 *  note:           This function overloads "osa.c"                             *
 *                                                                              *
 ********************************************************************************
 */

//------------------------------------------------------------------------------
#define _OS_Dtimer_Delete_DEFINED
//------------------------------------------------------------------------------

void _OS_Dtimer_Delete (OST_DTIMER *dtimer)
{

    OST_DTIMER *ptcb;

    ptcb = (OST_DTIMER*)&_OS_DTimers;
    while (ptcb->Flags.bNextEnable) {
        if (ptcb->Next == dtimer) {
            _OS_temp = (_OST_INT_TYPE)dtimer->Next;
            //------------------------------------------------------------------------------
            //ptcb->Next = _OS_temp;
            #if _BANKBITS_ == 2
                asm("   bcf     __status, 0                 ");
                asm("   btfsc   __OS_Dtimer_Delete$ptcb, 6   ");
                asm("   bsf     __status, 0                 ");
            #endif
            asm("   movf    __OS_Dtimer_Delete$ptcb, w       ");
            _OS_PIC12_AND_1F();
            asm("   movwf   __fsr                   ");
            asm("   incf    __fsr, f                ");
            //-------
            asm("   movf    __OS_temp, w  ");
            //-------
            #if _BANKBITS_ >= 1
                asm("   btfsc   __OS_Dtimer_Delete$ptcb, 5   ");
                asm("   bsf     __fsr, 5                    ");
            #endif
            #if _BANKBITS_ == 2
                asm("   btfsc   __status, 0 ");
                asm("   bsf     __fsr, 6            ");
            #endif
            asm("   movwf   __indf              ");
            _fsr = 0;

            if (!dtimer->Flags.bNextEnable) ptcb->Flags.bNextEnable = 0;
            dtimer->Flags.bActive = 0;
            return;
        }
        ptcb = ptcb->Next;
    }
}



//------------------------------------------------------------------------------
#endif  //  OS_ENABLE_DTIMERS
//------------------------------------------------------------------------------

































/************************************************************************************************
 *                                                                                              *
 *     M E S S A G E   Q U E U E S                                                              *
 *                                                                                              *
 ************************************************************************************************/


//------------------------------------------------------------------------------
#if defined(OS_ENABLE_QUEUE)
//------------------------------------------------------------------------------


/*
 ********************************************************************************
 *                                                                              *
 *  void _OS_Queue_Send (OST_QUEUE *pQueue, OST_MSG Msg)                        *
 *                                                                              *
 *------------------------------------------------------------------------------*
 *                                                                              *
 *  description:    (Internal function called by system kernel througth         *
 *                  service _OS_Queue_Send)                                     *
 *                                                                              *
 *                  Adds message into queue of pointers to messages. Deletes    *
 *                  first message if there is no free room to add new message.  *
 *                  Service OS_Queue_Send before adding new message checks for  *
 *                  free room. Thus messages will not deleted accidentally.     *
 *                                                                              *
 *                                                                              *
 *  parameters:     pQueue      - pointer to queue descriptor                   *
 *                  Msg        - pointer to  message to be added                *
 *                                                                              *
 *  on return:      OS_IsEventError() return 1, if first message was pushed out *
 *                                                                              *
 *  note:           This function overloads "osa.c"                             *
 *                                                                              *
 ********************************************************************************
 */

//------------------------------------------------------------------------------
#define _OS_Queue_Send_DEFINED
#define _OS_SQueue_Send_DEFINED
//------------------------------------------------------------------------------

void _OS_Queue_Send (OST_QUEUE *pQueue, OST_MSG Msg)
{
    OS_Flags.bEventError = 0;

    // if pQueue->Q.cSize == pQueue->Q.cFilled
    _fsr = (_FSR_TYPE) pQueue;
    asm("   movf    __indf, w               ");     // w = pQueue->Q.cSize
    asm("   incf    __fsr, f                ");     // fsr -> pQueue->Q.cFilled
    asm("   xorwf   __indf, w               ");     // Compare cSize  cFilled

    if (_status & 0x04) {   // Check ZERO flag

        _fsr = 0;
        OS_Flags.bEventError = 1;

        // pQueue->pMsg[pQueue->Q.cBegin] = Msg;
        _fsr = (_FSR_TYPE) pQueue;
        asm("   incf    __fsr, f        ");
        asm("   incf    __fsr, f        ");
        asm("   movf    __indf, w       ");     // w = pQueue->Q.cBegin
    PUT_MESSAGE:
        asm("   incf    __fsr, f        ");
        asm("   addwf   __indf, w       ");
        asm("   movwf   __fsr           ");     // fsr -> pQueue->pMsg[pQueue->Q.cBegin]



        #ifdef _16F59
            asm("   bcf     __status, 5     "); // Here we use ROM page bit to store FSR.7
            asm("   btfsc   __fsr, 7        ");
            asm("   bsf     __status, 5     ");
            asm("   bcf     __fsr, 7     ");
        #endif

        #if _BANKBITS_ == 2
            asm("   bcf     __status, 1     ");
            asm("   btfsc   __fsr, 6        ");
            asm("   bsf     __status, 1     ");
            asm("   bcf     __fsr, 6     ");
        #endif
        #if _BANKBITS_ >= 1
            asm("   bcf     __status, 0     ");
            asm("   btfsc   __fsr, 5        ");
            asm("   bsf     __status, 0     ");
            asm("   bcf     __fsr, 5     ");
        #endif

        asm("   movf    __OS_Queue_Send$Msg, w    ");

        #ifdef _16F59
            asm("   btfsc   __status, 5     "); // Here we use ROM page bit to store FSR.7
            asm("   bsf     __fsr, 7        ");
            asm("   dw  0x4A3   | (($ & 0x200) >>1)");
        #endif
        #if _BANKBITS_ == 2
            asm("   btfsc   __status, 1     ");
            asm("   bsf     __fsr, 6        ");
        #endif
        #if _BANKBITS_ >= 1
            asm("   btfsc   __status, 0     ");
            asm("   bsf     __fsr, 5        ");
        #endif
        asm("   movwf   __indf              ");


        // Set status, c = bEventError.
        // This will help to select register to increaze and compare to cSize
        // If there will be an error then cBegin++, else cFilled++

        _fsr = 0;
        _status &= 0xFE;
        if (OS_Flags.bEventError) _status |= 0x01;

        // pQueue->Q.cBegin++ (or pQueue->Q.cFileld++) and compare it to cSize
        _fsr = (_FSR_TYPE) pQueue;
        asm("   movf    __indf, w               ");
        asm("   btfsc   __status, 0             ");     // Skip ink FSR if there was not error
        asm("   incf    __fsr, f                ");
        asm("   incf    __fsr, f                ");     // fsr -> pQueue->Q.cBegin
        asm("   incf    __indf, f               ");     // cBegin++ ( cFilled++)
        if (_status & 1) {
            asm("   xorwf   __indf, w           ");     // Compare cBegin (or cFilled) to cSize
            asm("   btfsc   __status, 2         ");
            asm("   clrf    __indf              ");     // cBegin (or cFilled) = 0
        }
        asm("   clrf    __fsr                   ");
        return;
    }

    //temp = q.cBegin + q.cFilled;
    //if (temp >= q.cSize) temp -= q.cSize;
    asm("   incf    __fsr, f                ");
    asm("   movf    __indf, w               ");     // w = cBegin
    asm("   decf    __fsr, f                ");
    asm("   addwf   __indf, w               ");     // w += cFilled
    asm("   decf    __fsr, f                ");
    asm("   subwf   __indf, w               ");     // w = cSize - (cFileld + cBegin)
    asm("   btfss   __status, 0             ");     // > 0
    asm("   addwf   __indf, w               ");     // Correcting pointer
    asm("   subwf   __indf, w               ");     // w = cSize - (cFileld + cBegin)
    asm("   incf    __fsr, f                ");
    asm("   incf    __fsr, f                ");
    goto    PUT_MESSAGE;

}



/*
 ********************************************************************************
 *                                                                              *
 *  OST_MSG _OS_GetQueue (OST_QUEUE *pQueue)                                    *
 *                                                                              *
 *------------------------------------------------------------------------------*
 *                                                                              *
 *  description:    (Internal function called by system kernel througth         *
 *                  service OS_AcceptQueue)                                     *
 *                                                                              *
 *                  Get first pointer to message from queue. Before calling this*
 *                  function be sure that queue is not empty (OS_AcceptQueue does*
 *                  it automatically). After execution this function first      *
 *                  message will be deleted from queue.                         *
 *                                                                              *
 *  parameters:     pQueue      - pointer to queue descriptor                   *
 *                                                                              *
 *  on return:      first message from queue                                    *
 *                                                                              *
 *  note:           This function overloads "osa.c"                             *
 *                                                                              *
 ********************************************************************************
 */

//------------------------------------------------------------------------------
#define _OS_GetQueue_DEFINED
#define _OS_GetSQueue_DEFINED
//------------------------------------------------------------------------------

OST_MSG _OS_GetQueue (OST_QUEUE *pQueue)
{
    char temp;

    _fsr = (_FSR_TYPE)pQueue;
    _fsr++;
    _fsr++;
    asm("   movf    __indf, w           "); // w = cBegin
    asm("   incf    __fsr, f            "); // fsr -> pMsg
    asm("   addwf   __indf, w           ");
    asm("   movwf   __fsr               "); // fsr -> pMsg[cBegin]
    asm("   movf    __indf, w           "); // w = pMsg[cBegin]
    asm("   clrf    __fsr               ");
    asm("   movwf   __OS_GetQueue$temp   ");

    // Here FSR bits 5 and 6 are cleared (due to work with "temp")
    _fsr = (_FSR_TYPE)pQueue;
    asm("   movf    __indf, w           ");
    asm("   incf    __fsr, f            ");
    asm("   incf    __fsr, f            ");
    asm("   incf    __indf, f           "); // cBegin++
    asm("   subwf   __indf, w           ");
    asm("   btfsc   __status, 0         "); // if (cBegin >= cSize) cBegin = 0;
    asm("   clrf    __indf              ");
    asm("   decf    __fsr, f            ");
    asm("   decf    __indf, f           "); // cFilled--

    _fsr = 0;
    return (OST_MSG)temp;
}
#endif
//------------------------------------------------------------------------------

































